*
* Example of Gibbs sampling/conditional simulation.
*
open data nile.dat
calendar 1871
data(format=free,org=columns,skips=1) 1871:1 1970:1 nile
*
* This has two hyperparameters. h is the precision in the observation
* equation and psi is the relative variance in the drift in level.
*
compute nuh=1.0
compute s2h=100.0^2
compute hdraw=nuh/s2h
*
compute nupsi=1.0
compute s2psi=.1
compute psidraw=s2psi/nupsi
*
compute nburn =100
compute ndraws=2000
*
* These will keep track of the values of psi and h for the keeper draws.
*
set psigibbs  1 ndraws = 0.0
set hgibbs    1 ndraws = 0.0
*
* These will keep track of the sum and the sum of squares of the
* fitted series.
*
set fitsum   = 0.0
set fitsumsq = 0.0
*
infobox(action=define,progress,lower=-nburn,upper=ndraws) $
   "Gibbs Sampler"
do draw=-nburn,ndraws
   *
   * The key to analyzing the state space model using the Gibbs sampler
   * is DLM with type=csim. This does a draw from the joint distribution
   * of the measurement error and state disturbances subject to hitting
   * the observed data. The <<what>> and <<vhat>> have the simulated
   * values for those disturbances.
   *
   dlm(a=1.0,c=1.0,sv=1.0/hdraw,sw=psidraw/hdraw,y=nile,$
      type=csimulate,presample=diffuse,$
      what=what,vhat=vhat) / xstates
   *
   * Draw psi given h and the simulated w's. Because psi is a relative
   * *variance* (rather than precision), the %ranchisqr is in the
   * denominator
   *
   sstats / vhat(t)(1)^2>>sumvsq what(t)(1)^2>>sumwsq
   compute psidraw=(hdraw*sumwsq+nupsi*s2psi)/%ranchisqr(%nobs+nupsi)
   *
   * Draw h given psi and the simulated w's and v's
   *
   compute hdraw=$
      %ranchisqr(nuh+%nobs*2.0)/(nuh*s2h+sumvsq+sumwsq/psidraw)
   infobox(current=draw)
   if draw<=0
      next
   *
   * Do the bookkeeping here.
   *
   compute psigibbs(draw)=psidraw
   compute hgibbs(draw)=hdraw
   set fitsum   = fitsum+xstates(t)(1)
   set fitsumsq = fitsumsq+xstates(t)(1)^2
end do draw
infobox(action=remove)
*
* Compute statistics on the draws for h and psi
*
stats(fractiles) hgibbs
stats(fractiles) psigibbs
*
* Show the mean and upper and lower 90% CI (using a Normal approxi-
* mation) for the local level.
*
set fitsum   = fitsum/ndraws
set fitsumsq = fitsumsq/ndraws-fitsum^2
set lower    = fitsum+%invnormal(.05)*sqrt(fitsumsq)
set upper    = fitsum+%invnormal(.95)*sqrt(fitsumsq)
*
graph(footer="Local level and 90% CI from Gibbs sampling") 4
# nile
# fitsum
# lower / 3
# upper / 3

